/* --COPYRIGHT--,BSD_EX
 * Copyright (c) 2014, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 *
 *                       MSP430 CODE EXAMPLE DISCLAIMER
 *
 * MSP430 code examples are self-contained low-level programs that typically
 * demonstrate a single peripheral function or device feature in a highly
 * concise manner. For this the code may rely on the device's power-on default
 * register values and settings such as the clock configuration and care must
 * be taken when combining code from several examples to avoid potential side
 * effects. Also see www.ti.com/grace for a GUI- and www.ti.com/msp430ware
 * for an API functional library-approach to peripheral configuration.
 *
 * --/COPYRIGHT--*/
/*
 * Timers.c
 *
 *  Created on: May 25, 2020
 *      Author: JD Crutchfield
 */

#include <msp430.h>
#include "mcu.h"
#include "Timers.h"
#include "SD24.h"
#include "LCD.h"

unsigned long g_uiGlobalShutdownTimeout_ms = SHUTDOWN_TIMEOUT_MS;       //miliseconds between samples.  10ms=100sps

unsigned int g_uiTimeoutMilliseconds = 0;       //miliseconds between samples.  10ms=100sps
unsigned int g_uiPiezoTimeoutMilliseconds = 0;       //miliseconds between samples.  10ms=100sps


void InitTimers(void){
//Using ACLK so that we can wait in LPM3 for significant power savings
    //TA0 = PWM for Piezo buzzer on Boost_en pin
    TA0CCR0  = (ACLK_FREQ_HZ / PIEZO_PWM_FREQ_HZ) -1;
    TA0CCR2  = (ACLK_FREQ_HZ / PIEZO_PWM_FREQ_HZ) /2;    // 50% duty cycle
    TA0CCTL2 = OUTMOD_3;                                // CCR1 set/reset
    TA0CTL   = TASSEL_1;                                // SMCLK=DCO=16.384MHz, Not running

    //TA1 - generates a Interrupt every 1ms for timing and timeouts
    TA1CCTL0 = CCIE;                    // CCR0 Interrupt Enabled
    TA1CCR0  = 31;   // ~1ms period for tick
    TA1CTL   = TASSEL_1 | MC_1;         // SMCLK = DCO = 16.384MHz, Up Mode
}

void delay_ms(unsigned int milliseconds){
    //wait in LPM0 for fixed amount of milliseconds
    g_uiTimeoutMilliseconds = milliseconds;

    while(g_uiTimeoutMilliseconds){
        LPM3;
    }
}

void PiezoBuzzerEnable(unsigned int Timeout){
    //   Configure GPIO as timer PWM output.
    P2SEL0 |= BOOST_ENABLE_PIEZO_PWM;
    P2DIR  |= BOOST_ENABLE_PIEZO_PWM;

    //Start Timer - freq already configured in Timer Init
    TA0CTL |= TASSEL_1 | MC_1 | TACLR;

    //Set timerout for shutting off buzzer
    g_uiPiezoTimeoutMilliseconds = Timeout;
}

void PiezoBuzzerDisable(void){
    //   Configure GPIO back to boost EN control
    P2SEL0 &= ~BOOST_ENABLE_PIEZO_PWM;
    P2DIR  |= BOOST_ENABLE_PIEZO_PWM;

    //Stop Timer
    TA0CTL = TASSEL_2 | MC_0 | TACLR;
}




//TA1 Timeout ISR
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=TIMER1_A0_VECTOR
__interrupt void TA1_A0_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(TIMER1_A0_VECTOR))) TA1_ISR (void)
#else
#error Compiler not supported!
#endif
{
#ifndef     DEBUGGING_ANALOG_MEASUREMENT
    LCDStatemachineProcess();
#endif

    //Global Timeout counter for going into LPM4.5
    if(g_uiGlobalShutdownTimeout_ms){
        g_uiGlobalShutdownTimeout_ms--;
    }

    //Counter that decrements every millisecond for timing out of blocking wait loops.
    if(g_uiTimeoutMilliseconds){
        g_uiTimeoutMilliseconds--;

        if(g_uiTimeoutMilliseconds == 0){
            //Timeout.  Wake up device
            LPM3_EXIT;
        }
    }

    //Timeout for Piezo Buzzer
    if(g_uiPiezoTimeoutMilliseconds){
        g_uiPiezoTimeoutMilliseconds --;

        if(g_uiPiezoTimeoutMilliseconds == 0){
            PiezoBuzzerDisable();
        }
    }

    //Triggering Temperature measurments
    g_uiSD24SampleDelayMS--;

    //Start ADC Conversion
    if(g_uiSD24SampleDelayMS == 0){
        SD24CCTL3 |= SD24SC; //Start next ADC conversion
        g_uiSD24SampleDelayMS = SD24_SAMPLE_DELAY_MS;
    }
}









